// base
import React, { useCallback, useState } from 'react';

// components
import { CopyClipboard, PaginationTable, SearchBar, VideoModal } from 'components';

// layouts
import { MainLayout } from 'layouts';

// hooks
import { usePagination } from 'hooks';

// apis
import { videoAPI } from 'apis/video';

// types
import { ResponseVideo, SearchFormValue, VideoData } from 'types/video';

// consts
import { SWR_VIDEO_LIST, SWR_VIDEO_LIST_COUNT } from 'consts';

// services
import { LoopbackFilter } from 'services/loopback';

// libraries
import { ColumnProps } from 'antd/lib/table';
import { Button, Switch, Radio, message, Modal } from 'antd';
import { ExclamationCircleOutlined } from '@ant-design/icons';
import moment from 'moment';
import useSWR from 'swr';

// styles
import { Title, TableWrap, TopSection, CreateBtn } from './style';

export const VideoPage: React.FC = () => {
  const [typeValue, setTypeValue] = useState('전체');
  const [visible, setVisible] = useState(false);
  const [rowDetail, setRowDetail] = useState<ResponseVideo>({});
  const [isEdit, setIsEdit] = useState(false);
  const [filter, setFilter] = useState<LoopbackFilter>({
    where: {},
    skip: 0,
    limit: 10,
    order: 'created desc',
  });

  const {
    data: videoList,
    isValidating,
    mutate: videoListMutate,
  } = useSWR<ResponseVideo[]>(
    [SWR_VIDEO_LIST, JSON.stringify(filter)],
    (_, filter) => videoAPI.list(JSON.parse(filter)),
    {
      revalidateOnMount: true,
      revalidateOnFocus: false,
    }
  );

  const { data: count, mutate: videoCountMutate } = useSWR(
    [SWR_VIDEO_LIST_COUNT, JSON.stringify(filter)],
    (_, filter) => videoAPI.count(JSON.parse(filter).where),
    {
      revalidateOnMount: true,
      revalidateOnFocus: false,
    }
  );

  const { pagination, onChangePageSize } = usePagination({
    totalElement: count?.count ?? 0,
    onChangePagination: (page, pageSize) => {
      setFilter((prev) => {
        return {
          ...prev,
          skip: (page - 1) * pageSize,
          limit: pageSize,
        };
      });
    },
  });

  const handleSearch = useCallback((value: SearchFormValue) => {
    setFilter((prev) => {
      if (value.type === '전체') {
        return {
          ...prev,
          where: {
            or: [
              { ...value, type: '한화생명금융서비스' },
              { ...value, type: '한화라이프랩' },
              { ...value, type: 'GA' },
            ],
          },
        };
      } else {
        return {
          ...prev,
          where: {
            ...value,
          },
        };
      }
    });
  }, []);

  const handleReset = useCallback(() => {
    setFilter({
      skip: 0,
      limit: 10,
      order: 'created desc',
    });
  }, []);

  const handleCloseModal = () => {
    setIsEdit(false);
    setVisible(false);
    setRowDetail({});
  };

  const handleExposeChange = useCallback(
    async (value: boolean, id) => {
      await videoAPI.update(id, { expose: value });
      videoListMutate();
      videoCountMutate();
    },
    [videoListMutate, videoCountMutate]
  );

  const onTypeChange = (e: any) => {
    setTypeValue(e.target.value);
  };

  const handleEditVideo = (record: any) => {
    setRowDetail(record);
    setIsEdit(true);
    setVisible(true);
  };

  const onFinish = async (values: any) => {
    const { title, content, type, imgFileUrl, ctntUrl, totalTime, fileName, cdnCtntSeq } = values;
    const VideoData: VideoData = {
      title,
      content,
      type,
      imgFileUrl,
      ctntUrl,
      cdnCtntSeq,
      totalTime,
      fileName,
      status: 'active',
    };

    if (!isEdit) {
      try {
        await videoAPI.create(VideoData);
        videoListMutate();
        message.success('동영상이 등록되었습니다.');
      } catch (error) {
        message.error('동영상 등록에 실패하였습니다. 다시 시도해주세요.');
      }
    } else {
      try {
        if (rowDetail?.id) {
          await videoAPI.update(rowDetail.id, VideoData);
          videoListMutate();
          message.success('동영상이 수정되었습니다.');
        }
      } catch (error) {
        message.error('동영상 수정에 실패하였습니다. 다시 시도해주세요.');
      }
    }
    handleCloseModal();
  };

  const onDelete = async () => {
    Modal.confirm({
      title: '동영상을 삭제하시겠습니까?',
      icon: <ExclamationCircleOutlined />,
      okText: '삭제',
      cancelText: '취소',
      onOk: async () => {
        try {
          if (rowDetail?.id) {
            await videoAPI.update(rowDetail.id, { status: 'inactive' });
            videoListMutate();
            message.success('동영상이 삭제되었습니다.');
            handleCloseModal();
          }
        } catch (error) {
          message.error('삭제에 실패하였습니다. 다시 시도해주세요.');
        }
      },
    });
  };

  const onCopy = () => {
    message.success('URL 복사');
  };

  const handleVideoUpload = () => {
    const strWindowFeatures = 'resizable=yes,scrollbars=yes,status=yes,width=950px,height=800px';
    window.open(`${process.env.REACT_APP_VIDEO_UPLOAD}`, '동영상 업로드', strWindowFeatures);
  };

  const columns: ColumnProps<ResponseVideo>[] = [
    {
      title: '동영상 제목',
      dataIndex: 'title',
      key: 'title',
    },
    {
      title: '채널 구분',
      dataIndex: 'type',
      key: 'type',
    },
    {
      title: '등록일시',
      dataIndex: 'created',
      key: 'created',
      render: (value) => {
        return <>{moment(value).format('YYYY-MM-DD HH:mm:ss')}</>;
      },
    },
    {
      title: '노출여부',
      dataIndex: 'expose',
      key: 'expose',
      render: (text, record, idx) => {
        const status = [
          { txt: 'ON', status: true },
          { txt: 'OFF', status: false },
        ];
        return (
          <Switch
            checkedChildren={status[0].txt}
            unCheckedChildren={status[1].txt}
            checked={record.expose}
            onChange={(e) => handleExposeChange(e, record.id)}
          />
        );
      },
    },
    {
      title: '상세보기',
      render: (record) => {
        return (
          <Button
            onClick={(e) => {
              e.preventDefault();
              handleEditVideo(record);
            }}
          >
            상세보기
          </Button>
        );
      },
    },
    {
      title: 'URL 복사',
      render: (text, record) => {
        return (
          <CopyClipboard
            text={`${process.env.REACT_APP_FO}/?detail=${record.id}`}
            onCopy={onCopy}
            children={<Button>복사하기</Button>}
          />
        );
      },
    },
  ];

  return (
    <MainLayout>
      <TopSection>
        <Title>동영상 조회&#38;관리</Title>
        <CreateBtn
          type="primary"
          onClick={() => {
            setIsEdit(false);
            setVisible(true);
          }}
        >
          동영상 등록
        </CreateBtn>
      </TopSection>
      <SearchBar
        inputItems={[
          {
            key: 'title',
            name: 'title',
            title: '동영상 제목',
            placeholder: '동영상 제목을 입력해주세요',
          },
        ]}
        searchDate={{ title: '등록 일자', visible: true }}
        customFormItems={[
          {
            key: 'type',
            name: 'type',
            title: '게시 채널 구분',
            render: () => {
              return (
                <Radio.Group onChange={onTypeChange} value={typeValue}>
                  <Radio value="전체">전체</Radio>
                  <Radio value="한화생명금융서비스">한화생명금융서비스</Radio>
                  <Radio value="한화라이프랩">한화라이프랩</Radio>
                  <Radio value="GA">GA</Radio>
                </Radio.Group>
              );
            },
          },
        ]}
        handleSearch={handleSearch}
        onReset={handleReset}
      />
      <TableWrap>
        <PaginationTable
          customRight={
            <Button type="primary" onClick={handleVideoUpload}>
              동영상 업로드
            </Button>
          }
          rowKey="id"
          loading={isValidating}
          dataSource={videoList ?? []}
          pagination={pagination}
          onChangePageSize={onChangePageSize}
          columns={columns}
        />
      </TableWrap>
      {visible && (
        <VideoModal
          visible={visible}
          initialValues={rowDetail}
          isEdit={isEdit}
          handleCancel={handleCloseModal}
          onSubmit={onFinish}
          onDelete={onDelete}
        />
      )}
    </MainLayout>
  );
};
