// base
import React, { useCallback, useState } from 'react';

// component
import { PaginationTable, SearchBar } from 'components';

// hooks
import { usePagination } from 'hooks';

// apis
import { usersDataUploadAPI } from 'apis/log_usersDataUpload';

// const
import { SWR_USERS_DATA_LIST, SWR_USERS_DATA_LIST_COUNT } from 'consts';

// types
import { ResponseUsersDataUpload } from 'types/usersData';

// services
import { LoopbackFilter } from 'services/loopback';

// style
import { ContentWrap, TableWrap } from './style';

// libraries
import { ColumnProps } from 'antd/lib/table';
import useSWR from 'swr';
import moment from 'moment';

export const UsersDataList: React.FC = () => {
  const [filter, setFilter] = useState<LoopbackFilter>({
    where: {},
    skip: 0,
    limit: 10,
    order: 'created desc, id desc',
  });

  const { data: usersDataUploadList, isValidating } = useSWR(
    [SWR_USERS_DATA_LIST, JSON.stringify(filter)],
    (_, filter) => usersDataUploadAPI.get(JSON.parse(filter)),
    {
      revalidateOnMount: true,
      revalidateOnFocus: false,
    }
  );

  const { data: count } = useSWR(
    [SWR_USERS_DATA_LIST_COUNT, JSON.stringify(filter)],
    (_, filter) => usersDataUploadAPI.count(JSON.parse(filter).where),
    {
      revalidateOnMount: true,
      revalidateOnFocus: false,
    }
  );

  const { pagination, onChangePageSize } = usePagination({
    totalElement: count?.count ?? 0,
    onChangePagination: (page, pageSize) => {
      setFilter((prev) => {
        return {
          ...prev,
          skip: (page - 1) * pageSize,
          limit: pageSize,
        };
      });
    },
  });

  const handleSearch = useCallback((value: { [key: string]: any }) => {
    setFilter((prev) => {
      return {
        ...prev,
        where: {
          ...value,
        },
      };
    });
  }, []);

  const handleReset = useCallback(() => {
    setFilter({ where: {}, skip: 0, limit: 10, order: 'created desc, id desc' });
  }, []);

  const columns: ColumnProps<ResponseUsersDataUpload>[] = [
    {
      title: '파일명',
      dataIndex: 'fileName',
      key: 'fileName',
    },
    {
      title: '등록일시',
      dataIndex: 'created',
      key: 'created',
      render: (value) => {
        return <>{moment(value).format('YYYY-MM-DD HH:mm:ss')}</>;
      },
    },
  ];

  return (
    <ContentWrap>
      <SearchBar
        searchDate={{ title: '등록 일자', visible: true }}
        handleSearch={handleSearch}
        onReset={handleReset}
      />
      <TableWrap>
        <PaginationTable
          loading={isValidating}
          rowKey="id"
          dataSource={usersDataUploadList ?? []}
          columns={columns}
          pagination={pagination}
          onChangePageSize={onChangePageSize}
        />
      </TableWrap>
    </ContentWrap>
  );
};
