//base
import React, { useCallback, useEffect, useState } from 'react';

//component
import { PageLoader, PaginationTable, SearchBar } from 'components';

//apis
import { statisticsAPI } from 'apis/statistics';

//hooks
import { usePagination } from 'hooks';

//utils
import { SWR_STATISTICS } from 'consts';
import { columnsHead, columnsArea, columnsRegion, columnsBranch } from './columns';
import { downloadExcel } from 'libs/utils';

//libraries
import { Button, Radio, Menu } from 'antd';
import useSWR from 'swr';
import moment from 'moment';

//style
import { StatisticsListWrap, TableWrap, ListMenuWrap } from './style';
import { LoopbackFilter } from 'services/loopback';

export const StatisticsList: React.FC = () => {
  const [loading, setLoading] = useState(true);

  const [typeValue, setTypeValue] = useState('전체');
  const [listType, setListType] = useState('l2_name');
  const [columns, setColumns] = useState(columnsHead);
  const [filter, setFilter] = useState<LoopbackFilter>({});

  const { data, error } = useSWR(
    [SWR_STATISTICS, JSON.stringify(filter), listType],
    (_, filter, listType) => statisticsAPI.action(JSON.parse(filter), listType)
  );
  if (data && loading) {
    setLoading(false);
  }
  if (error) console.error(error);

  useEffect(() => {
    setLoading(true);
  }, [filter]);

  const handleSearch = useCallback((value) => {
    const and = value.created
      ? [
          { created: { gte: value.created && value.created.gte.toDate() } },
          { created: { lte: value.created && value.created.lte.toDate() } },
        ]
      : [];

    if (value.type === '전체') {
      setFilter({
        where: {
          and: and,
          employee_id: value.fp,
        },
      });
    } else {
      setFilter({
        where: {
          and: and,
          type: value.type,
        },
      });
    }
  }, []);

  const handleReset = useCallback(() => {
    setFilter({});
    setTypeValue('전체');
  }, []);

  const onTypeChange = (e: any) => {
    setTypeValue(e.target.value);
  };

  const handleListMenu = (menu: string) => {
    setLoading(true);
    switch (menu) {
      case 'l2_name':
        setColumns(columnsHead);
        break;
      case 'region':
        setColumns(columnsRegion);
        break;
      case 'area':
        setColumns(columnsArea);
        break;
      case 'branch':
        setColumns(columnsBranch);
        break;
    }
    setListType(menu);
  };

  const { pagination, onChangePageSize } = usePagination({
    totalElement: data ? data.length : 0,
  });

  const handleXlsxDownload = () => {
    if (!data) {
      alert('다운로드할 데이터가 없습니다!');
      return;
    }

    const xlsxData: any[] = [];
    switch (listType) {
      case 'l2_name':
        xlsxData.push([
          '',
          '채널 구분',
          '사업본부',
          '영상조회수',
          'PDF조회수',
          '메인페이지 접속 수',
        ]);
        break;
      case 'region':
        xlsxData.push(['', '사업본부', '권역', '영상조회수', 'PDF조회수', '메인페이지 접속 수']);
        break;
      case 'area':
        xlsxData.push([
          '',
          '사업본부',
          '권역',
          '지역단',
          '영상조회수',
          'PDF조회수',
          '메인페이지 접속 수',
        ]);
        break;
      case 'branch':
        xlsxData.push([
          '',
          '사업본부',
          '권역',
          '지역단',
          '지점(기관)',
          '영상조회수',
          'PDF조회수',
          '메인페이지 접속 수',
        ]);
        break;
    }

    data.forEach((file: any, itemIndex: number) => {
      const idx = data.length - itemIndex;

      switch (listType) {
        case 'l2_name':
          xlsxData.push([idx, file.title, file.l2_name, file.video, file.document, file.main]);
          break;
        case 'region':
          xlsxData.push([idx, file.l2_name, file.region, file.video, file.document, file.main]);
          break;
        case 'area':
          xlsxData.push([
            idx,
            file.l2_name,
            file.region,
            file.area,
            file.video,
            file.document,
            file.main,
          ]);
          break;
        case 'branch':
          xlsxData.push([
            idx,
            file.l2_name,
            file.region,
            file.area,
            file.branch,
            file.video,
            file.document,
            file.main,
          ]);
          break;
      }
    });

    downloadExcel(xlsxData, ['sheet1'], `${moment().format('YYYY-MM-DD')} 통계 요약`);
  };

  return (
    <StatisticsListWrap>
      {loading && <PageLoader />}
      <SearchBar
        searchDate={{ title: '검색일자', visible: true }}
        handleSearch={handleSearch}
        noSearchFromToday={true}
        onReset={handleReset}
        customFormItems={[
          {
            key: 'type',
            name: 'type',
            title: '채널 구분',
            render: () => {
              return (
                <Radio.Group onChange={onTypeChange} value={typeValue}>
                  <Radio value="전체">전체</Radio>
                  <Radio value="한화생명금융서비스">한화생명금융서비스</Radio>
                  <Radio value="한화라이프랩">한화라이프랩</Radio>
                  <Radio value="GA">GA</Radio>
                </Radio.Group>
              );
            },
          },
        ]}
      />
      <ListMenuWrap
        mode="horizontal"
        defaultSelectedKeys={[`${listType}`]}
        selectedKeys={[`${listType}`]}
        onClick={(clickItem) => {
          handleListMenu(clickItem.key);
        }}
      >
        <Menu.Item key={'l2_name'}>사업본부 기준</Menu.Item>
        <Menu.Item key={'region'}>권역 기준</Menu.Item>
        <Menu.Item key={'area'}>지역단 기준</Menu.Item>
        <Menu.Item key={'branch'}>지점(기관) 기준</Menu.Item>
      </ListMenuWrap>
      <TableWrap>
        <PaginationTable
          noAsync
          rowKey="id"
          customLeft={<Button onClick={handleXlsxDownload}>Excel 다운로드</Button>}
          dataSource={data || []}
          columns={columns}
          pagination={pagination}
          onChangePageSize={onChangePageSize}
        />
      </TableWrap>
    </StatisticsListWrap>
  );
};
