//base
import React, { useCallback, useState } from 'react';

// components
import { CopyClipboard, DocumentModal, PaginationTable, SearchBar } from 'components';

// layout
import { MainLayout } from 'layouts';

// hooks
import { usePagination } from 'hooks';

// apis
import { documentAPI } from 'apis/document';

// types
import { ResponseDocument, DocumentData, SearchFormValue } from 'types/document';

// const
import { SWR_DOCUMENT_LIST, SWR_DOCUMENT_LIST_COUNT } from 'consts';

// services
import { LoopbackFilter } from 'services/loopback';

//libraries
import { UploadFile } from 'antd/lib/upload/interface';
import { ColumnProps } from 'antd/lib/table';
import { Button, message, Modal, Radio, Switch } from 'antd';
import { ExclamationCircleOutlined } from '@ant-design/icons';
import moment from 'moment';
import useSWR from 'swr';

// style
import { Title, TopSection, CreateBtn, TableWrap } from './style';

export const DocumentPage: React.FC = () => {
  const [modalVisible, setModalVisible] = useState(false);
  const [selectedRowData, setSelectedRowData] = useState<ResponseDocument>();
  const [isEdit, setIsEdit] = useState(false);
  const [file, setFile] = useState<UploadFile<File>>();
  const [typeValue, setTypeValue] = useState('전체');
  const [filter, setFilter] = useState<LoopbackFilter>({
    where: {},
    skip: 0,
    limit: 10,
    order: 'created desc',
  });

  const {
    data: documentList,
    isValidating,
    mutate: listMutate,
  } = useSWR(
    [SWR_DOCUMENT_LIST, JSON.stringify(filter)],
    (_, filter) => documentAPI.get(JSON.parse(filter)),
    {
      revalidateOnMount: true,
      revalidateOnFocus: false,
    }
  );

  const { data: count, mutate: countMutate } = useSWR(
    [SWR_DOCUMENT_LIST_COUNT, JSON.stringify(filter)],
    (_, filter) => documentAPI.count(JSON.parse(filter).where),
    {
      revalidateOnMount: true,
      revalidateOnFocus: false,
    }
  );

  const { pagination, onChangePageSize } = usePagination({
    totalElement: count?.count ?? 0,
    onChangePagination: (page, pageSize) => {
      setFilter((prev) => {
        return {
          ...prev,
          skip: (page - 1) * pageSize,
          limit: pageSize,
        };
      });
    },
  });

  const handleSearch = useCallback((value: SearchFormValue) => {
    setFilter((prev) => {
      if (value.target === '전체') {
        return {
          ...prev,
          where: {
            or: [{ target: '한화생명금융서비스' }, { target: '한화라이프랩' }, { target: 'GA' }],
          },
        };
      } else {
        return {
          ...prev,
          where: {
            ...value,
          },
        };
      }
    });
  }, []);

  const handleReset = useCallback(() => {
    setFilter({
      where: {},
      skip: 0,
      limit: 10,
      order: 'created desc',
    });
  }, []);

  const handleCloseModal = () => {
    setIsEdit(false);
    setModalVisible(false);
  };

  const handleEditRowData = (record: ResponseDocument) => {
    setSelectedRowData(record);
    setIsEdit(true);
    setModalVisible(true);
  };

  const onSubmitForm = async (values: DocumentData) => {
    const { title, target } = values;

    const DocumentData: DocumentData = {
      title: title,
      target: target,
      status: 'active',
      exposure: true,
      fileName: file?.name,
    };

    try {
      if (!isEdit) {
        if (!file) {
          message.error('PDF 등록해주세요.');
          return;
        }
        await documentAPI.uploadFile(DocumentData, file);
        message.success('컨텐츠가 등록되었습니다.');
        return;
      }
      if (selectedRowData?.id) {
        DocumentData.id = selectedRowData?.id;
        await documentAPI.uploadFile(DocumentData, file);

        message.success('컨텐츠가 수정되었습니다.');
      }
    } catch (error) {
      message.error('컨텐츠 등록에 실패하였습니다. 다시 시도해주세요.');
    } finally {
      listMutate();
      countMutate();
      handleCloseModal();
    }
  };

  const onDeleteProduct = (id: string) => {
    Modal.confirm({
      title: '해당 상품 정보를 삭제하시겠습니까?',
      icon: <ExclamationCircleOutlined />,
      okText: '삭제',
      cancelText: '취소',
      onOk: async () => {
        try {
          await documentAPI.update(id, { status: 'inactive' });
          listMutate();
          countMutate();
          message.success('PDF가 삭제되었습니다.');
          handleCloseModal();
        } catch (error) {
          message.error('삭제에 실패하였습니다. 다시 시도해주세요.');
        }
      },
    });
  };

  const handleExposeChange = useCallback(
    async (value: boolean, id) => {
      await documentAPI.update(id, {
        exposure: value,
      });
      listMutate();
      countMutate();
    },
    [listMutate, countMutate]
  );

  const handleOpenCreateModal = () => {
    setIsEdit(false);
    setModalVisible(true);
  };

  const onTypeChange = (e: any) => {
    setTypeValue(e.target.value);
  };

  const onCopy = () => {
    message.success('URL 복사');
  };

  const colums: ColumnProps<any>[] = [
    {
      title: 'PDF 제목',
      dataIndex: 'title',
      key: 'title',
    },
    {
      title: '채널 구분',
      dataIndex: 'target',
      key: 'target',
    },
    {
      title: '등록일시',
      dataIndex: 'created',
      key: 'created',
      render: (value) => {
        return <>{moment(value).format('YYYY-MM-DD HH:mm:ss')}</>;
      },
    },
    {
      title: '노출여부',
      dataIndex: 'exposure',
      key: 'exposure',
      render: (text, record) => {
        const status = [
          { txt: 'ON', status: true },
          { txt: 'OFF', status: false },
        ];
        return (
          <Switch
            checkedChildren={status[0].txt}
            unCheckedChildren={status[1].txt}
            checked={record.exposure}
            onChange={(e) => handleExposeChange(e, record.id)}
          />
        );
      },
    },
    {
      title: '상세보기',
      render: (record) => {
        return (
          <Button
            onClick={() => {
              handleEditRowData(record);
            }}
          >
            상세 보기
          </Button>
        );
      },
    },
    {
      title: 'URL 복사',
      render: (text, record) => {
        return (
          <CopyClipboard
            text={`${process.env.REACT_APP_FO}/docs/detail?id=${record.id}`}
            onCopy={onCopy}
            children={<Button>복사하기</Button>}
          />
        );
      },
    },
  ];

  return (
    <MainLayout>
      <TopSection>
        <Title>PDF 조회&#38;관리</Title>
        <CreateBtn type="primary" onClick={handleOpenCreateModal}>
          PDF 등록
        </CreateBtn>
      </TopSection>
      <SearchBar
        inputItems={[
          {
            key: 'title',
            name: 'title',
            title: 'PDF 제목',
            placeholder: '제목을 입력해주세요',
          },
        ]}
        searchDate={{ title: '등록 일자', visible: true }}
        customFormItems={[
          {
            key: 'target',
            name: 'target',
            title: '게시 채널 구분',
            render: () => {
              return (
                <Radio.Group onChange={onTypeChange} value={typeValue}>
                  <Radio value="전체">전체</Radio>
                  <Radio value="한화생명금융서비스">한화생명금융서비스</Radio>
                  <Radio value="한화라이프랩">한화라이프랩</Radio>
                  <Radio value="GA">GA</Radio>
                </Radio.Group>
              );
            },
          },
        ]}
        handleSearch={handleSearch}
        onReset={handleReset}
      />
      <TableWrap>
        <PaginationTable
          loading={isValidating}
          rowKey="id"
          dataSource={documentList ?? []}
          pagination={pagination}
          onChangePageSize={onChangePageSize}
          columns={colums}
        />
      </TableWrap>
      <DocumentModal
        visible={modalVisible}
        initialValues={selectedRowData}
        isEdit={isEdit}
        onCancel={handleCloseModal}
        onSubmit={onSubmitForm}
        setFile={setFile}
        onDeleteProduct={onDeleteProduct}
      />
    </MainLayout>
  );
};
