// base
import React, { useMemo, useEffect, useRef } from 'react';

// libraries
import { Table, Button, Select, Row, Col, Space } from 'antd';
import { TableProps, ColumnsType } from 'antd/lib/table';
import { PaginationProps } from 'antd/lib/pagination';

// styles
import { TableWrap } from './style';

// defines
const DEFAULT_PAGE_SIZE = 10;
const pageSizeRange = [10, 20, 50, 100];

interface PaginationTableProps<T> extends TableProps<T> {
  noAsync?: boolean;
  noIndex?: boolean;
  columns: ColumnsType<T>;
  dataSource: T[];
  customLeft?: React.ReactNode;
  customRight?: React.ReactNode;
  onChangePageSize?: (page: number, pageSize: number) => void;
  onDownload?: () => void;
}

interface Pagination extends PaginationProps {
  total: number;
  current: number;
  pageSize: number;
  showSizeChanger: boolean;
}

export const PaginationTable = <T extends {}>(props: PaginationTableProps<T>) => {
  const {
    noAsync = false,
    noIndex = false,
    columns,
    dataSource,
    customLeft,
    customRight,
    onChangePageSize,
    onDownload,
    ...tableOptions
  } = props;

  const pagination = useMemo<Pagination>(() => {
    return {
      total: 0,
      current: 1,
      pageSize: DEFAULT_PAGE_SIZE,
      showSizeChanger: false,
      position: ['bottomCenter'],
      ...tableOptions.pagination,
    };
  }, [tableOptions.pagination]);

  const handleChangePageSize = (pageSize: number) => {
    if (onChangePageSize) {
      onChangePageSize(1, pageSize);
    }
  };

  const dataSourceRef = useRef(dataSource);

  useEffect(() => {
    if (dataSourceRef.current.length === dataSource.length) {
      return;
    }

    if (pagination.total && dataSource.length === 0) {
      if (pagination.onChange) {
        pagination.onChange(
          pagination.current > 1 ? pagination.current - 1 : 1,
          pagination.pageSize
        );
      }
    }
  }, [pagination, dataSource]);

  useEffect(() => {
    dataSourceRef.current = dataSource;
  }, [dataSource]);

  return (
    <TableWrap>
      <div className="top-section">
        <div className="left-section">
          {pagination.total !== null && (
            <p className="result-text">
              검색 결과{' '}
              <span className="total">
                총 &#40;<span className="num">{pagination.total ? pagination.total : 0}</span>
                개&#41;
              </span>
            </p>
          )}
          {customLeft}
        </div>

        <Row>
          {onDownload && (
            <Col>
              <Button type="primary" onClick={onDownload}>
                전체 다운로드
              </Button>
            </Col>
          )}
          <Col>
            <Space size={5}>
              {customRight}
              <Select
                style={{ width: 100 }}
                defaultValue={DEFAULT_PAGE_SIZE}
                value={pagination.pageSize ? pagination.pageSize : undefined}
                onChange={handleChangePageSize}
              >
                {pageSizeRange.map((size) => (
                  <Select.Option key={size} value={size}>
                    {size}개씩
                  </Select.Option>
                ))}
              </Select>
            </Space>
          </Col>
        </Row>
      </div>
      <Table
        {...tableOptions}
        pagination={pagination}
        columns={
          noIndex
            ? columns
            : [
                {
                  title: '순번',
                  dataIndex: 'index',
                  key: 'index',
                  width: 80,
                },
                ...columns,
              ]
        }
        dataSource={
          !noAsync && dataSource.length
            ? dataSource.map((item, index) => {
                return {
                  ...item,
                  index:
                    pagination.total -
                    index -
                    (pagination.current * pagination.pageSize - pagination.pageSize),
                };
              })
            : dataSource.map((item, index) => ({
                ...item,
                index: pagination.total - index,
              }))
        }
        bordered
      />
    </TableWrap>
  );
};
