// base
import React, { useEffect, useState } from 'react';

// components
import { CopyClipboard } from 'components';

// types
import { DocumentData, ResponseDocument } from 'types/document';

// libraries
import { Button, Descriptions, Form, Input, message, Radio, Upload } from 'antd';

// styles
import { ModalWrap, ConfirmBtn, DeleteBtn } from './style';

interface DocumentModalProps {
  visible: boolean;
  initialValues?: ResponseDocument;
  isEdit: boolean;
  onSubmit: (values: DocumentData) => void;
  onCancel: () => void;
  setFile: (values: any) => void;
  onDeleteProduct: (id: string) => void;
}

export const DocumentModal: React.FC<DocumentModalProps> = ({
  visible,
  initialValues,
  isEdit,
  onSubmit,
  onCancel,
  setFile,
  onDeleteProduct,
}) => {
  const [form] = Form.useForm();
  const [fileName, setFileName] = useState('');
  const accessList = [{ key: '한화생명금융서비스' }, { key: '한화라이프랩' }, { key: 'GA' }];
  const accessType = Object.entries(accessList).map(([key, value]) => {
    return {
      label: value.key,
      value: value.key,
    };
  });

  const onCopy = () => {
    message.success('URL 복사');
  };

  useEffect(() => {
    form.setFieldsValue({
      title: '',
      target: '',
      upload: '',
      file: '',
    });
    setFileName('');

    if (isEdit && initialValues?.title) {
      form.setFieldsValue({
        title: initialValues?.title,
      });
    }
    if (isEdit && initialValues?.target) {
      form.setFieldsValue({
        target: initialValues?.target,
      });
    }
    if (isEdit && initialValues?.fileName) {
      setFileName(initialValues?.fileName);
    }
  }, [visible, initialValues, isEdit, form]);

  return (
    <ModalWrap
      centered
      title={isEdit ? 'PDF 상세 ' : 'PDF  등록'}
      visible={visible}
      footer={null}
      onCancel={onCancel}
      width={800}
      forceRender
    >
      <div className="sub-title">
        <h3>정보 확인</h3>
        {isEdit && (
          <DeleteBtn onClick={() => onDeleteProduct(initialValues?.id ?? '')}>PDF 삭제</DeleteBtn>
        )}
      </div>

      <Form
        name="documentForm"
        form={form}
        initialValues={initialValues ? initialValues : undefined}
        onFinish={onSubmit}
      >
        <Descriptions bordered column={12}>
          <Descriptions.Item label="채널 구분" span={12}>
            <Form.Item name="target" rules={[{ required: true, message: '필수 항목입니다.' }]}>
              <Radio.Group
                options={accessType}
                onChange={(e) => {
                  form.setFieldsValue({
                    target: e.target.value,
                  });
                }}
              />
            </Form.Item>
          </Descriptions.Item>
          <Descriptions.Item label="PDF 제목" span={12} labelStyle={{ width: 120 }}>
            <Form.Item name="title" rules={[{ required: true, message: '필수 항목입니다.' }]}>
              <Input placeholder={'PDF 제목을 입력해주세요'} />
            </Form.Item>
          </Descriptions.Item>
          <Descriptions.Item label="업로드" span={12}>
            <Form.Item name="upload">
              <Upload
                fileList={[]}
                name="file"
                maxCount={1}
                multiple={false}
                showUploadList={false}
                beforeUpload={() => {
                  return false;
                }}
                onChange={(info) => {
                  setFile(info.file);
                  setFileName(info.fileList[0].name);
                }}
                accept=".pdf"
              >
                <Button>PDF 파일 업로드</Button>
              </Upload>
            </Form.Item>
            <div className="fileName">{fileName}</div>
          </Descriptions.Item>
          {isEdit && (
            <Descriptions.Item label="URL">
              <CopyClipboard
                text={`${process.env.REACT_APP_FO}/docs/detail?id=${initialValues?.id}`}
                onCopy={onCopy}
                children={
                  <Button type="link">{`${process.env.REACT_APP_FO}/docs/detail?id=${initialValues?.id}`}</Button>
                }
              />
            </Descriptions.Item>
          )}
        </Descriptions>
        <ConfirmBtn htmlType="submit" type="primary">
          확인
        </ConfirmBtn>
      </Form>
    </ModalWrap>
  );
};
