// base
import React, { useCallback, useState } from 'react';

// store
import { authProfileSelector } from 'store/selector';

// apis
import { authAPI } from 'apis/auth';

// services
import { STORAGE_SESSION_ACT } from 'services/storage';

// routes
import { ROUTE_ROOT } from 'routes/const';

// types
import { UpdatePassword, UpdateUser } from 'types/auth';

// libraries
import { Layout, Button, Modal, Descriptions, Form, Input, message } from 'antd';
import { UserOutlined, SettingOutlined, ExclamationCircleOutlined } from '@ant-design/icons';
import { Link, useNavigate } from 'react-router-dom';
import { useRecoilState } from 'recoil';
import { useForm } from 'antd/es/form/Form';

// style
import { HeaderBox } from './style';

// assets
import logo from '../../assets/images/logo.png';

type ValidateStatuses = 'success' | 'warning' | 'error' | 'validating' | '';

export const Header: React.FC = () => {
  const [profile, setProfile] = useRecoilState(authProfileSelector);
  const navigate = useNavigate();
  const [updateModalVisible, setUpdateModalVisible] = useState(false);
  const [updatePasswordModalVisible, setUpdatePasswordModalVisible] = useState(false);
  const [form] = useForm();
  const [passwordForm] = useForm();
  const [checkPassword, setCheckPassword] = useState<ValidateStatuses>('');

  const sessionClear = () => {
    sessionStorage.removeItem(STORAGE_SESSION_ACT);
    setProfile({
      created: '',
      updated: '',
      username: '',
      role: 'MASTER',
      status: 'inactive',
      team: '',
      fp_id: '',
      menu: '[]',
      email: '',
      id: '',
    });
    navigate(ROUTE_ROOT);
  };

  const onLogout = async () => {
    await authAPI.logout();
    sessionClear();
  };

  const onUpdate = async ({ id, update }: { id: string; update: UpdateUser }) => {
    try {
      const act = sessionStorage.getItem(STORAGE_SESSION_ACT);
      if (act) {
        await authAPI.patch(id, update);
        const profile = await authAPI.profile(act);
        setProfile(profile);
        onUpdateModal();
        message.success('계정 정보가 변경되었습니다.');
      }
    } catch (error) {
      message.error('계정 정보 수정에 실패하였습니다. 관리자에게 문의해주세요.');
    }
  };

  const onUpdatePassword = async (data: UpdatePassword) => {
    try {
      await authAPI.updatePassword(data);
      sessionClear();
      message.success('비밀번호가 변경되었습니다. \n 다시 로그인해주세요.');
      setCheckPassword('');
    } catch (error) {
      setCheckPassword('error');
      message.error('비밀번호를 확인해주세요.');
    }
  };

  const onUpdateModal = useCallback(() => {
    setUpdateModalVisible(!updateModalVisible);
  }, [updateModalVisible]);

  const onUpdatePasswordModal = useCallback(() => {
    setUpdatePasswordModalVisible(!updatePasswordModalVisible);
  }, [updatePasswordModalVisible]);

  const updatePasswordConfirm = () => {
    Modal.confirm({
      title: '계정 정보 수정 시 로그아웃됩니다.',
      icon: <ExclamationCircleOutlined />,
      content: '수정하시겠습니까?',
      okText: '수정',
      cancelText: '취소',
      onOk: () => passwordForm.submit(),
    });
  };

  if (!profile.id) {
    return <div />;
  }

  return (
    <HeaderBox>
      <Layout.Header className="header-wrap">
        <Link to={'/'}>
          <h1>
            <img src={logo} alt="한화생명" />
          </h1>
        </Link>
        <div className="header-user">
          <div className="header-user-wrap">
            <UserOutlined />
            <div className="header-user-txt">
              {`${profile.team} / ${profile.username}`}
              <SettingOutlined
                style={{ marginLeft: 20, marginRight: 20 }}
                onClick={onUpdateModal}
              />
              <Button onClick={onLogout}>LOGOUT</Button>
            </div>
          </div>
        </div>
        {updateModalVisible && (
          <Modal
            onCancel={onUpdateModal}
            title="계정 관리"
            visible={updateModalVisible}
            footer={
              <div style={{ textAlign: 'center' }}>
                <Button type="primary" size="large" onClick={() => form.submit()}>
                  저장
                </Button>
              </div>
            }
          >
            <Form
              form={form}
              initialValues={{ team: profile.team, username: profile.username }}
              onFinish={(values) => {
                onUpdate({
                  id: profile.id,
                  update: values,
                });
              }}
            >
              <Descriptions
                title="계정 정보"
                bordered
                column={24}
                extra={<Button onClick={onUpdatePasswordModal}>비밀번호 재설정</Button>}
              >
                <Descriptions.Item label="ID" span={24}>
                  {profile.email}
                </Descriptions.Item>
                <Descriptions.Item label="소속" span={24}>
                  <Form.Item name="team" noStyle>
                    <Input />
                  </Form.Item>
                </Descriptions.Item>
                <Descriptions.Item label="이름" span={24}>
                  <Form.Item name="username" noStyle>
                    <Input />
                  </Form.Item>
                </Descriptions.Item>
              </Descriptions>
            </Form>
          </Modal>
        )}

        <Modal
          onCancel={onUpdatePasswordModal}
          title="비밀번호 재설정"
          visible={updatePasswordModalVisible}
          footer={
            <div style={{ textAlign: 'center' }}>
              <Button type="primary" size="large" onClick={updatePasswordConfirm}>
                저장
              </Button>
            </div>
          }
        >
          <Form
            form={passwordForm}
            onFinish={(values) => {
              onUpdatePassword({
                email: profile.email,
                oriPassword: values.oriPassword,
                password: values.password,
              });
            }}
          >
            <Descriptions title="계정 정보" bordered column={24}>
              <Descriptions.Item
                label={
                  <span>
                    현재 비밀번호<span style={{ color: 'red' }}>*</span>
                  </span>
                }
                span={24}
              >
                <Form.Item
                  name="oriPassword"
                  style={{ marginBottom: 0 }}
                  validateStatus={checkPassword}
                  help={checkPassword ? '비밀번호를 확인해주세요.' : undefined}
                  rules={[{ required: true, message: '비밀번호를 입력해주세요' }]}
                >
                  <Input.Password onFocus={() => setCheckPassword('')} />
                </Form.Item>
              </Descriptions.Item>
              <Descriptions.Item
                label={
                  <span>
                    신규 비밀번호<span style={{ color: 'red' }}>*</span>
                  </span>
                }
                span={24}
              >
                <Form.Item
                  name="password"
                  rules={[
                    { required: true, message: '비밀번호를 입력해주세요' },
                    () => ({
                      validator(_, value) {
                        const reg = /^(?=.*[A-Za-z])(?=.*\d)[A-Za-z\d]{8,50}$/;

                        if (!value || reg.test(value)) {
                          return Promise.resolve();
                        }
                        return Promise.reject(new Error('비밀번호 조합을 확인해주세요.'));
                      },
                    }),
                  ]}
                  style={{ marginBottom: 0 }}
                  hasFeedback
                >
                  <Input.Password />
                </Form.Item>
              </Descriptions.Item>
              <Descriptions.Item
                label={
                  <span>
                    비밀번호 재확인<span style={{ color: 'red' }}>*</span>
                  </span>
                }
                span={24}
              >
                <Form.Item
                  name="passwordCheck"
                  style={{ marginBottom: 0 }}
                  dependencies={['password']}
                  rules={[
                    { required: true, message: '비밀번호를 입력해주세요' },
                    ({ getFieldValue }) => ({
                      validator(_, value) {
                        console.log(value);

                        if (!value || getFieldValue('password') === value) {
                          return Promise.resolve();
                        }
                        return Promise.reject(new Error('신규 비밀번호와 매칭되지 않습니다.'));
                      },
                    }),
                  ]}
                  hasFeedback
                >
                  <Input.Password />
                </Form.Item>
              </Descriptions.Item>
            </Descriptions>
            <span style={{ color: '#959595', display: 'block', marginTop: 10 }}>
              * 비밀번호는 8자 이상의 영문자, 숫자의 조합으로 설정해주세요.
            </span>
          </Form>
        </Modal>
      </Layout.Header>
    </HeaderBox>
  );
};
