// Copyright IBM Corp. 2016,2019. All Rights Reserved.
// Node module: loopback-workspace
// This file is licensed under the MIT License.
// License text available at https://opensource.org/licenses/MIT

"use strict";
require("dotenv").config();
const loopback = require("loopback");
const boot = require("loopback-boot");
const cors = require("cors");
const bodyParserErrorHandler = require("express-body-parser-error-handler");

const app = (module.exports = loopback());

app.use(bodyParserErrorHandler());

app.start = function () {
  // start the web server
  return app.listen(process.env.API_PORT, function () {
    app.emit("started");
    const baseUrl = app.get("url").replace(/\/$/, "");
    console.log("Web server listening at: %s", baseUrl);
    if (app.get("loopback-component-explorer")) {
      const explorerPath = app.get("loopback-component-explorer").mountPath;
      console.log("Browse your REST API at %s%s", baseUrl, explorerPath);
    }
  });
};

// Bootstrap the application, configure models, datasources and middleware.
// Sub-apps like REST API are mounted via boot scripts.
boot(app, __dirname, function (err) {
  if (err) throw err;

  app.use(
    cors({
      origin: (origin, cb) => {
        const IS_DEV = process.env.IS_DEV;
        if (IS_DEV === "yes") return cb(null, true);

        const WHITE_LIST = process.env.WHITE_LIST.split(",");
        const isWhitelisted = WHITE_LIST.indexOf(origin) !== -1;
        if (isWhitelisted) {
          cb(null, isWhitelisted);
        } else {
          cb(new Error("no whitelist"));
        }
      },
    })
  );

  app.use(
    loopback.token({
      model: app.models.accessToken,
    })
  );

  // start the server if `$ node server.js`
  if (require.main === module) app.start();
});
