//base
import React, { useCallback, useState } from 'react';

//components
import { DataUploader, PageLoader } from 'components';

//apis
import { usersDataAPI } from 'apis/usersData';
import { usersDataUploadAPI } from 'apis/log_usersDataUpload';

//utils
import { downloadExcel } from 'libs/utils';

//style
import { ContentWrap } from './style';

//libraries
import { UploadFile } from 'antd/lib/upload/interface';
import { message, Button, Modal } from 'antd';
import { ExclamationCircleOutlined } from '@ant-design/icons';

export const UsersDataUpload: React.FC = () => {
  const [file, setFile] = useState<UploadFile<File>>();
  const [loading, setLoading] = useState(false);

  const handleFormReset = () => {
    setFile(undefined);
  };

  const handleFormSubmit = useCallback(async () => {
    if (file) {
      try {
        setLoading(true);
        const upload = await usersDataAPI.uploadFile(file);
        if (!upload.fileName) throw upload;
        await usersDataUploadAPI.create(upload);
        message.success('데이터가 업로드 되었습니다.');
        handleFormReset();
      } catch (err) {
        message.error('데이터 업로드에 실패했습니다. 다시 시도해주세요.');
      } finally {
        setLoading(false);
      }
    }
  }, [file]);

  const onClickBasicExcelDownload = () => {
    const columns = [
      [
        'L1_NAME',
        'L1_CODE',
        'L2_NAME',
        'L2_CODE',
        '권역',
        '권역코드',
        '지역단',
        '지역단코드',
        '지점',
        '지점코드',
        '사원번호',
        '성명',
      ],
    ];
    const sheetName = ['한화생명금융서비스', '한화라이프랩', 'GA'];

    downloadExcel(columns, sheetName, '조직데이터양식');
  };

  const onDeleteUsersData = () => {
    Modal.confirm({
      title: '기존 조직 데이터를 삭제하시겠습니까?',
      icon: <ExclamationCircleOutlined />,
      okText: '삭제',
      cancelText: '취소',
      onOk: async () => {
        try {
          await usersDataAPI.deleteAll();
          message.success('모든 조직 데이터가 삭제되었습니다.');
        } catch (error) {
          message.error('삭제에 실패하였습니다. 다시 시도해주세요.');
        }
      },
    });
  };

  return (
    <>
      {loading && <PageLoader />}
      <ContentWrap>
        <div>
          <div className="delete-data">
            <Button onClick={onDeleteUsersData}>조직데이터 삭제</Button>
            <p>
              * 기존 조직 데이터를 삭제하는 경우, 새로운 데이터가 업로드 될 때까지 서비스 이용에
              장애가 발생할 수 있습니다.
            </p>
          </div>
          <Button onClick={onClickBasicExcelDownload} className="download-excel">
            양식 다운로드
          </Button>
        </div>
        <DataUploader
          file={file}
          setFile={setFile}
          onFinish={handleFormSubmit}
          onReset={handleFormReset}
        />
      </ContentWrap>
    </>
  );
};
