//base
import React, { useCallback, useMemo, useState } from 'react';

//component
import { PaginationTable, SearchBar, PageLoader } from 'components';

//apis
import { logActionsAPI } from 'apis/log_actions';

//utils
import { SWR_LOGACTIONS, SWR_LOGACTIONS_COUNT } from 'consts';
import { downloadExcel } from 'libs/utils';

//libraries
import { Button, Input, Radio } from 'antd';
import useSWR from 'swr';
import moment from 'moment';

//style
import { StatisticsDetailWrap, TableWrap } from './style';
import { usePagination } from 'hooks';
import { LoopbackFilter } from 'services/loopback';

export const StatisticsOrigin: React.FC = () => {
  const [filter, setFilter] = useState<LoopbackFilter>({
    where: {
      and: [
        { created: { gte: moment().startOf('days') } },
        { created: { lte: moment().endOf('days') } },
      ],
    },
    skip: 0,
    limit: 10,
    order: 'created desc',
  });

  const [searchFp, setSearchFp] = useState('');
  const [loading, setLoading] = useState(false);

  const { data: logData, isValidating } = useSWR(
    [SWR_LOGACTIONS, JSON.stringify(filter)],
    (_, filter) => logActionsAPI.action(JSON.parse(filter)),
    {
      revalidateOnMount: true,
      revalidateOnFocus: false,
    }
  );

  const { data: count } = useSWR(
    [SWR_LOGACTIONS_COUNT, JSON.stringify(filter)],
    (_, filter) => logActionsAPI.count(JSON.parse(filter).where),
    {
      revalidateOnMount: true,
      revalidateOnFocus: false,
    }
  );

  const { pagination, onChangePageSize } = usePagination({
    totalElement: count?.count ?? 0,
    onChangePagination: (page, pageSize) => {
      setFilter((prev) => {
        return {
          ...prev,
          skip: (page - 1) * pageSize,
          limit: pageSize,
        };
      });
    },
  });

  const processLogData = (data: any) => {
    return data.map((log: any) => {
      return {
        ...log,
        action:
          log.action === 'main' ? '메인페이지' : log.action === 'document' ? 'PDF문서' : '영상',
        logDate: moment(log.created).format('YYYY-MM-DD'),
        logTime: moment(log.created).format('HH:mm:ss'),
      };
    });
  };

  const handleSearch = useCallback((value) => {
    if (value.created) {
      const dateDiff = value.created.lte.diff(value.created.gte, 'days');
      if (dateDiff > 30) {
        alert('최대 31일까지 조회가능합니다. 기간을 다시 설정해주세요.');
        return;
      }
    }

    console.log(value);

    const and = value.created
      ? [
          { created: { gte: value.created && value.created.gte.toDate() } },
          { created: { lte: value.created && value.created.lte.toDate() } },
        ]
      : [
          { created: { gte: moment().startOf('days') } },
          { created: { lte: moment().endOf('days') } },
        ];
    const or = [{ action: value.logType }];

    setFilter({
      where: {
        and: and,
        or: or,
        employee_id: value.fp,
      },
      skip: 0,
      limit: 10,
      order: 'created desc',
    });
  }, []);

  const handleReset = useCallback(() => {
    setFilter({
      where: {
        and: [
          { created: { gte: moment().startOf('days') } },
          { created: { lte: moment().endOf('days') } },
        ],
      },
      skip: 0,
      limit: 10,
      order: 'created desc',
    });
  }, []);

  const handleXlsxDownload = useCallback(async () => {
    setLoading(true);

    const fileData = await logActionsAPI.action({
      where: filter.where,
      order: 'created desc',
    });

    if (!fileData) {
      alert('다운로드할 데이터가 없습니다!');
      return;
    }

    const xlsxData: any[] = [
      [
        '',
        '접속일자',
        '접속시간',
        '사업본부명',
        '권역',
        '지역단',
        '지점(기관)',
        'FP코드',
        'FP이름',
        'log',
      ],
    ];

    fileData.forEach((file: any, itemIndex: number) => {
      const idx = fileData.length - itemIndex;
      xlsxData.push([
        idx,
        moment(file.created).format('YYYY-MM-DD'),
        moment(file.created).format('HH:mm:ss'),
        file.l2_name,
        file.region,
        file.area,
        file.branch,
        file.employee_id,
        file.name,
        file.action === 'main' ? '메인페이지' : file.action === 'document' ? 'PDF문서' : '영상',
      ]);
    });
    setLoading(false);

    downloadExcel(xlsxData, ['sheet1'], `${moment().format('YYYY-MM-DD')} 통계원본데이터`, true);
  }, [filter]);

  const options = useMemo(() => {
    return [
      { label: '메인페이지', value: 'main' },
      { label: 'PDF문서', value: 'document' },
      { label: '영상', value: 'video' },
    ];
  }, []);

  const columns: any[] = useMemo(() => {
    return [
      {
        title: '접속일자',
        dataIndex: 'logDate',
        key: 'logDate',
      },
      {
        title: '접속시간',
        dataIndex: 'logTime',
        key: 'logTime',
      },
      {
        title: '사업본부',
        dataIndex: 'l2_name',
        key: 'l2_name',
      },
      {
        title: '권역',
        dataIndex: 'region',
        key: 'region',
      },
      {
        title: '지역단',
        dataIndex: 'area',
        key: 'area',
      },
      {
        title: '지점(기관)',
        dataIndex: 'branch',
        key: 'branch',
      },
      {
        title: 'FP 고유번호',
        dataIndex: 'employee_id',
        key: 'employee_id',
      },
      {
        title: 'FP명',
        dataIndex: 'name',
        key: 'name',
      },
      {
        title: 'log',
        dataIndex: 'action',
        key: 'action',
      },
    ];
  }, []);

  return (
    <StatisticsDetailWrap>
      {loading && <PageLoader />}
      <div style={{ marginBottom: '10px', color: 'red' }}>
        ※ 통계 원본데이터는 최대 31일까지 조회가 가능합니다.
      </div>
      <SearchBar
        searchDate={{ title: '검색일자', visible: true }}
        periodOfOneMonth={true}
        handleSearch={handleSearch}
        onReset={handleReset}
        customFormItems={[
          {
            key: 'fp',
            name: 'fp',
            title: 'FP 고유번호',
            render: () => (
              <Input
                value={searchFp}
                onChange={(e) => {
                  setSearchFp(e.target.value);
                }}
              />
            ),
          },
          {
            key: 'logType',
            name: 'logType',
            title: '필터',
            render: () => {
              return <Radio.Group options={options} />;
            },
          },
        ]}
      />
      <TableWrap>
        <PaginationTable
          loading={isValidating}
          rowKey="id"
          customLeft={<Button onClick={() => handleXlsxDownload()}>Excel 다운로드</Button>}
          dataSource={logData ? processLogData(logData) : []}
          columns={columns}
          pagination={pagination}
          onChangePageSize={onChangePageSize}
        />
      </TableWrap>
    </StatisticsDetailWrap>
  );
};
