// base
import React, { useCallback } from 'react';

// components
import { SearchDatePicker } from './SearchDatePicker';

// libraries
import { Button, Form, FormInstance, Input } from 'antd';

// types
import { CustomFormItem, SearchDateType, SearchInputType } from './SearchBarTypes';

// style
import { Wrap } from './style';

interface SearchProps {
  inputItems?: SearchInputType[];
  customForm?: FormInstance;
  customFormItems?: CustomFormItem[];
  searchDate?: SearchDateType;
  onReset?: () => void;
  handleSearch?(values: { [key: string]: any }): void;
  noSearchFromToday?: boolean;
  periodOfOneMonth?: boolean;
}

export const SearchBar: React.FC<SearchProps> = ({
  inputItems,
  searchDate,
  customFormItems,
  customForm,
  onReset,
  handleSearch,
  noSearchFromToday,
  periodOfOneMonth,
}) => {
  const [form] = Form.useForm();

  const handleReset = useCallback(() => {
    form.resetFields();

    if (onReset) {
      onReset();
    }
  }, [onReset, form]);

  return (
    <Wrap>
      <Form
        initialValues={{ type: '전체', target: '전체', logType: 'main' }}
        key="searchForm"
        form={customForm ? customForm : form}
        onFinish={(value) => {
          if (handleSearch) {
            if (value.dateRange && value.dateRange[0] && value.dateRange[1]) {
              const startAt = value.dateRange[0];
              const endAt = value.dateRange[1];
              value['created'] = { gte: startAt, lte: endAt };
              delete value.dateRange;
            }

            Object.keys(value).forEach((searchKey) => {
              const noApplicateArr = ['type', 'created', 'target', 'fp'];
              if (!value[searchKey]) {
                value[searchKey] = undefined;
                return;
              }
              if (noApplicateArr.includes(searchKey)) {
                return;
              }
              value[searchKey] = { like: value[searchKey] };
            });

            handleSearch(value);
          }
        }}
      >
        {inputItems &&
          inputItems.map((item) => {
            return (
              <div className="search-row" key={item.key}>
                <p className="search-title">{item.title}</p>
                <Form.Item name={item.name}>
                  <Input placeholder={item.placeholder} />
                </Form.Item>
              </div>
            );
          })}
        {searchDate?.visible && (
          <div className="search-row">
            <p className="search-title">{searchDate.title}</p>
            <Form.Item name="dateRange">
              <SearchDatePicker
                noSearchFromToday={noSearchFromToday}
                periodOfOneMonth={periodOfOneMonth}
              />
            </Form.Item>
          </div>
        )}
        {customFormItems &&
          customFormItems.map((item, index) => (
            <div className="search-row" key={index}>
              <p className="search-title">{item.title}</p>
              <Form.Item name={item.name}>{item.render(form)}</Form.Item>
            </div>
          ))}
        <Form.Item>
          <div className="button-row">
            <Button htmlType="submit" type="primary">
              검색
            </Button>
            <Button onClick={handleReset}>초기화</Button>
          </div>
        </Form.Item>
      </Form>
    </Wrap>
  );
};
