// base
import React from 'react';

// libraries
import moment from 'moment';
import { DatePicker, Row, Col, Button } from 'antd';

type MomentDateRange = [moment.Moment, moment.Moment];

interface SearchDatePickerProps {
  value?: MomentDateRange;
  onChange?: (value: MomentDateRange) => void;
  recentTwoWeek?: boolean;
  recentThreeDay?: boolean;
  noSearchFromToday?: boolean;
  periodOfOneMonth?: boolean;
}

export const SearchDatePicker: React.FC<SearchDatePickerProps> = ({
  value,
  onChange,
  recentThreeDay,
  recentTwoWeek,
  noSearchFromToday,
  periodOfOneMonth,
}) => {
  const handleChange = (values: MomentDateRange) => {
    if (periodOfOneMonth) {
      const dateDiff = values[1].endOf('days').diff(values[0].startOf('days'), 'days');
      if (dateDiff > 30) {
        alert('최대 31일까지 조회가능합니다. 기간을 다시 설정해주세요.');
      }
    }
    if (onChange) {
      onChange([values[0].startOf('days'), values[1].endOf('days')]);
    }
  };

  const handleChangeShortcut = (key: string) => {
    const started = moment().add(-1, 'days').startOf('days');
    const ended = moment().add(-1, 'days').endOf('days');

    if (onChange) {
      switch (key) {
        case 'RECENT_THREE_DAYS':
          started.subtract(2, 'days');
          break;

        case 'RECENT_WEEK':
          started.subtract(7, 'days');
          break;

        case 'RECENT_TWO_WEEK':
          started.subtract(15, 'days');
          break;

        case 'RECENT_MONTH':
          started.subtract(30, 'days');
          break;

        default:
          break;
      }

      onChange([started, ended]);
    }
  };

  const disabledDate = (current: any) => {
    if (noSearchFromToday) {
      return current && current > moment().startOf('day');
    }
    return false;
  };

  return (
    <Row gutter={10}>
      <Col>
        <DatePicker.RangePicker
          disabledDate={disabledDate}
          value={value}
          onChange={(values) => handleChange(values as MomentDateRange)}
        />
      </Col>

      {!noSearchFromToday && (
        <Col>
          <Button onClick={() => handleChangeShortcut('TODAY')}>오늘</Button>
        </Col>
      )}

      {recentThreeDay && (
        <Col>
          <Button onClick={() => handleChangeShortcut('RECENT_THREE_DAYS')}>최근 3일</Button>
        </Col>
      )}

      <Col>
        <Button onClick={() => handleChangeShortcut('RECENT_WEEK')}>최근 7일</Button>
      </Col>
      {recentTwoWeek && (
        <Col>
          <Button onClick={() => handleChangeShortcut('RECENT_TWO_WEEK')}>최근 14일</Button>
        </Col>
      )}

      <Col>
        <Button onClick={() => handleChangeShortcut('RECENT_MONTH')}>최근 30일</Button>
      </Col>
    </Row>
  );
};
