// base
import React, { useCallback, useEffect, useState } from 'react';

// components
import { Editor, PaginationTable, SearchBar, CopyClipboard } from 'components';

// apis
import { cdnurlsAPI } from 'apis/cdnurls';

// types
import { ResponseVideo, VideoData } from 'types/video';
import { ResponseCdnurls } from 'types/cdnurls';

// hooks
import { usePagination } from 'hooks';
import { cdnurlsSelector } from 'store/selector';

// consts
import { SWR_CDNURLS_LIST } from 'consts';

// services
import { LoopbackFilter } from 'services/loopback';

// libraries
import { Button, Form, Input, Radio, Descriptions, Modal, message } from 'antd';
import { ColumnProps } from 'antd/lib/table';
import useSWR from 'swr';
import { useSetRecoilState } from 'recoil';

// styles
import { ConfirmBtn, ModalWrap, TableWrap, UploadBtn } from './style';

interface VideoModalProps {
  visible: boolean;
  initialValues: ResponseVideo;
  isEdit: boolean;
  onSubmit: (values: VideoData) => void;
  onDelete: () => void;
  handleCancel: () => void;
}

export const VideoModal: React.FC<VideoModalProps> = ({
  visible,
  initialValues,
  isEdit,
  onSubmit,
  onDelete,
  handleCancel,
}) => {
  const [form] = Form.useForm();
  const [videoListVisible, setVideoListVisible] = useState(false);
  const [videoListData, setVideoListData] = useState<ResponseCdnurls>({});
  const [fileName, setFileName] = useState('');
  const setCdnurls = useSetRecoilState(cdnurlsSelector);
  const [cdnCtntSeq, setCdnCtntSeq] = useState<string[]>([]);

  const [filter, setFilter] = useState<LoopbackFilter>({
    where: {},
    order: 'created desc',
  });

  const { data: cdnurlsList, isValidating } = useSWR<ResponseCdnurls[]>(
    [SWR_CDNURLS_LIST, JSON.stringify(filter)],
    (_, filter) => cdnurlsAPI.cdnurls(JSON.parse(filter)),
    {
      revalidateOnMount: true,
      revalidateOnFocus: false,
    }
  );

  const { pagination, onChangePageSize } = usePagination({
    totalElement: cdnurlsList?.length ?? 0,
  });

  const handleSearch = useCallback((value: { [key: string]: any }) => {
    setFilter((prev) => {
      return {
        ...prev,
        where: {
          ...value,
        },
      };
    });
  }, []);

  const handleReset = useCallback(() => {
    setFilter({
      where: {},
      order: 'created desc',
    });
  }, []);

  const onChangeContent = (content: string) => {
    form.setFieldsValue({
      name: 'content',
      value: content,
    });
  };

  const showVideoListModal = () => {
    setVideoListVisible(true);
  };

  const handleVideoListOk = () => {
    setVideoListVisible(false);
  };

  const handleVideoCancel = () => {
    setVideoListVisible(false);
  };

  const columns: ColumnProps<any>[] = [
    {
      title: '썸네일',
      dataIndex: 'imgFileUrl',
      key: 'imgFileUrl',
      render: (text: string) => <img className="thumbnail" src={text} alt="대표이미지" />,
    },
    {
      title: '원본 파일명',
      dataIndex: 'fileName',
      key: 'fileName',
    },
  ];

  const onCopy = () => {
    message.success('URL 복사');
  };

  useEffect(() => {
    setCdnurls(cdnurlsList ?? []);
    setCdnCtntSeq([`${initialValues.cdnCtntSeq}`]);
  }, [setCdnurls, cdnurlsList, setCdnCtntSeq, initialValues.cdnCtntSeq]);

  useEffect(() => {
    form.setFieldsValue({
      created: '',
      type: '',
      title: '',
      content: '',
      imgFileUrl: '',
      ctntUurl: '',
      totalTime: '',
      fileName: '',
    });

    if (isEdit && initialValues?.created) {
      form.setFieldsValue({
        created: initialValues?.created,
      });
    }

    if (isEdit && initialValues?.type) {
      form.setFieldsValue({
        type: initialValues?.type,
      });
    }

    if (isEdit && initialValues?.title) {
      form.setFieldsValue({
        title: initialValues?.title,
      });
    }

    if (isEdit && initialValues?.content) {
      form.setFieldsValue({
        content: initialValues?.content,
      });
    }

    if (isEdit && initialValues?.imgFileUrl) {
      form.setFieldsValue({
        imgFileUrl: initialValues?.imgFileUrl,
      });
    }

    if (isEdit && initialValues?.ctntUrl) {
      form.setFieldsValue({
        ctntUrl: initialValues?.ctntUrl,
      });
    }

    if (isEdit && initialValues?.totalTime) {
      form.setFieldsValue({
        totalTime: initialValues?.totalTime,
      });
    }

    if (isEdit && initialValues?.fileName) {
      form.setFieldsValue({
        fileName: initialValues?.fileName,
      });
    }
  }, [initialValues, isEdit, form]);

  return (
    <Modal
      title={isEdit ? '동영상 상세' : '동영상 등록'}
      centered
      visible={visible}
      onCancel={handleCancel}
      width={688}
      maskClosable={true}
      destroyOnClose
      footer={false}
      closable={true}
    >
      {visible && (
        <ModalWrap>
          <Form
            name="vieoForm"
            form={form}
            initialValues={initialValues ? initialValues : undefined}
            onFinish={(values) => {
              onSubmit({
                ...values,
                ...videoListData,
              });
            }}
          >
            <Descriptions
              title={
                <div style={{ fontSize: 18 }}>
                  정보 확인
                  {isEdit ? (
                    <Button className="deleteBtn" onClick={onDelete}>
                      동영상 삭제
                    </Button>
                  ) : (
                    <></>
                  )}
                </div>
              }
              bordered
              column={12}
            >
              <Descriptions.Item label="채널 구분" span={12} labelStyle={{ width: 150 }}>
                <Form.Item
                  name="type"
                  style={{ marginBottom: 0 }}
                  rules={[{ required: true, message: '필수 항목입니다.' }]}
                >
                  <Radio.Group>
                    <Radio value="한화생명금융서비스">한화생명금융서비스</Radio>
                    <Radio value="한화라이프랩">한화라이프랩</Radio>
                    <Radio value="GA">GA</Radio>
                  </Radio.Group>
                </Form.Item>
              </Descriptions.Item>
              <Descriptions.Item label="동영상 제목" span={12}>
                <Form.Item
                  name="title"
                  style={{ marginBottom: 0 }}
                  rules={[{ required: true, message: '필수 항목입니다.' }]}
                >
                  <Input
                    type="text"
                    placeholder="동영상 제목을 입력해주세요"
                    style={{ resize: 'none', width: '100%' }}
                  />
                </Form.Item>
              </Descriptions.Item>
              <Descriptions.Item label="내용" span={12}>
                <Form.Item
                  name="content"
                  style={{ marginBottom: 0 }}
                  rules={[{ required: true, message: '필수 항목입니다.' }]}
                >
                  <Editor
                    onChange={(content) => {
                      if (content) {
                        onChangeContent(content);
                      }
                    }}
                  />
                </Form.Item>
              </Descriptions.Item>
              <Descriptions.Item label="영상선택" span={12}>
                <div className="upload-box">
                  <UploadBtn onClick={showVideoListModal}>동영상 선택</UploadBtn>
                  {(fileName || (initialValues && initialValues.fileName)) && (
                    <div className="file-name">{fileName || initialValues.fileName}</div>
                  )}
                </div>
                <Form.Item className="upload" name="upload" style={{ marginBottom: 0 }}>
                  <Modal
                    title="동영상 선택"
                    visible={videoListVisible}
                    width={900}
                    footer={false}
                    centered
                    onCancel={handleVideoCancel}
                  >
                    <Descriptions
                      title={<div style={{ fontSize: 18 }}>동영상 목록</div>}
                    ></Descriptions>
                    <SearchBar
                      inputItems={[
                        {
                          key: 'fileName',
                          name: 'fileName',
                          title: '원본 파일명',
                          placeholder: '파일명을 입력해주세요',
                        },
                      ]}
                      handleSearch={handleSearch}
                      onReset={handleReset}
                    />
                    <TableWrap>
                      <PaginationTable
                        rowSelection={{
                          type: 'radio',
                          selectedRowKeys: cdnCtntSeq,
                          onSelect: function (record) {
                            setVideoListData(record);
                            setFileName(record.fileName);
                          },
                          onChange: function (record) {
                            setCdnCtntSeq(record as string[]);
                          },
                        }}
                        noAsync
                        rowKey="cdnCtntSeq"
                        loading={isValidating}
                        dataSource={cdnurlsList ?? []}
                        pagination={pagination}
                        onChangePageSize={onChangePageSize}
                        columns={columns}
                      />
                    </TableWrap>
                    <ConfirmBtn type="primary" onClick={handleVideoListOk}>
                      확인
                    </ConfirmBtn>
                  </Modal>
                </Form.Item>
              </Descriptions.Item>
              {isEdit && (
                <Descriptions.Item label="URL">
                  <CopyClipboard
                    text={`${process.env.REACT_APP_FO}/?detail=${initialValues.id}`}
                    onCopy={onCopy}
                    children={
                      <Button type="link">{`${process.env.REACT_APP_FO}/?detail=${initialValues.id}`}</Button>
                    }
                  />
                </Descriptions.Item>
              )}
            </Descriptions>
            <ConfirmBtn type="primary" htmlType="submit">
              {isEdit ? '저장' : '등록'}
            </ConfirmBtn>
          </Form>
        </ModalWrap>
      )}
    </Modal>
  );
};
