//base
import React, { useCallback, useState } from 'react';

//component
import { PaginationTable, SearchBar, PageLoader } from 'components';

//apis
import { statisticsAPI } from 'apis/statistics';

//utils
import { SWR_STATISTICS_DETAIL } from 'consts';
import { downloadExcel } from 'libs/utils';

//libraries
import { Button, Input, message, Radio, Select } from 'antd';
import useSWR from 'swr';
import moment from 'moment';

//type
import { StatisticsDetailValue } from 'types/statistics';

//style
import { StatisticsDetailWrap, TableWrap, SelectWrap } from './style';
import { usePagination } from 'hooks';
import { LoopbackFilter } from 'services/loopback';

const { Option } = Select;

export const StatisticsDetail: React.FC = () => {
  const [filter, setFilter] = useState<LoopbackFilter>({});
  const [typeValue, setTypeValue] = useState('전체');
  const [selectWholeData, setSelectWholeData] = useState<any>();
  const [selectList, setSelectList] = useState<any>({
    l2_name: [],
    region: [],
    area: [],
    branch: [],
  });
  const [selectValue, setSelectValue] = useState<any>({});
  const [searchFp, setSearchFp] = useState('');
  const [loading, setLoading] = useState(false);

  const { data, isValidating } = useSWR(
    [SWR_STATISTICS_DETAIL, JSON.stringify(filter)],
    (_, filter) => statisticsAPI.detail(JSON.parse(filter)),
    {
      revalidateOnMount: true,
      revalidateOnFocus: false,
    }
  );

  const { pagination, onChangePageSize } = usePagination({
    totalElement: data?.length ?? 0,
  });

  const handleSearch = useCallback(
    (value) => {
      const and = value.created
        ? [
            { created: { gte: value.created && value.created.gte.toDate() } },
            { created: { lte: value.created && value.created.lte.toDate() } },
          ]
        : [];

      if (value.type === '전체') {
        setFilter({
          where: {
            and: and,
            employee_id: value.fp,
          },
        });
      } else {
        setFilter({
          where: {
            and: and,
            type: value.type,

            l2_name: selectValue.l2_name,
            region: selectValue.region,
            area: selectValue.area,
            branch: selectValue.branch,
            employee_id: value.fp,
          },
        });
      }
    },
    [selectValue]
  );

  const handleReset = useCallback(() => {
    setFilter({});
    setTypeValue('전체');
    setSelectValue({});
  }, []);

  const handleXlsxDownload = useCallback(async () => {
    setLoading(true);
    const data = await statisticsAPI.detail({
      ...filter,
    });

    if (!data) {
      alert('다운로드할 데이터가 없습니다!');
      return;
    }

    const xlsxData: any[] = [
      [
        '',
        '사업본부명',
        '권역',
        '지역단',
        '지점(기관)',
        'FP코드',
        'FP이름',
        '영상조회수',
        'PDF조회수',
        '메인페이지 접속 수',
      ],
    ];

    data.forEach((file: StatisticsDetailValue, itemIndex: number) => {
      const idx = data.length - itemIndex;
      xlsxData.push([
        idx,
        file.l2_name,
        file.region,
        file.area,
        file.branch,
        file.employee_id,
        file.name,
        file.video,
        file.document,
        file.main,
      ]);
    });
    setLoading(false);

    downloadExcel(xlsxData, ['sheet1'], `${moment().format('YYYY-MM-DD')} 통계`);
  }, [filter]);

  const onTypeChange = async (e: any) => {
    setTypeValue(e.target.value);
    setSelectValue({});
    try {
      const value = await statisticsAPI.select({ title: e.target.value });
      setSelectWholeData(value);
      const list = Array.from(new Set(value.map((item: any) => item.l2_name)));
      setSelectList({ l2_name: list, region: [], area: [], branch: [] });
    } catch (error) {
      message.error('데이터를 불러오는데에 실패했습니다. 다시 시도해주세요.');
    }
  };

  const setOptionData = (option: string) => {
    return (
      selectList[option] &&
      selectList[option].map((option: string, idx: number) => (
        <Option value={option} key={idx}>
          {option}
        </Option>
      ))
    );
  };

  const nextSelect = (txt: string) => {
    switch (txt) {
      case 'l2_name':
        return 'region';
      case 'region':
        return 'area';
      case 'area':
        return 'branch';
    }
  };

  const onSelectChange = (e: string, title: string) => {
    if (!selectWholeData) {
      return;
    }

    if (title === 'branch') {
      setSelectValue({ ...selectValue, branch: e });
      return;
    }

    const list = selectWholeData.map((item: any) => {
      if (item[title] === e) {
        const next: any = nextSelect(title);
        return item[next];
      }

      return null;
    });
    const _list = Array.from(new Set(list)).filter((x) => x !== null);

    switch (title) {
      case 'l2_name':
        setSelectValue({ l2_name: e });
        setSelectList({ ...selectList, region: _list, area: [], branch: [] });
        break;
      case 'region':
        setSelectValue({ l2_name: selectValue.l2_name, region: e });
        setSelectList({ ...selectList, area: _list, branch: [] });
        break;
      case 'area':
        setSelectValue({ l2_name: selectValue.l2_name, region: selectValue.region, area: e });
        setSelectList({ ...selectList, branch: _list });
        break;
    }
  };

  const columns: any[] = [
    {
      title: '사업본부',
      dataIndex: 'l2_name',
      key: 'l2_name',
    },
    {
      title: '권역',
      dataIndex: 'region',
      key: 'region',
    },
    {
      title: '지역단',
      dataIndex: 'area',
      key: 'area',
    },
    {
      title: '지점(기관)',
      dataIndex: 'branch',
      key: 'branch',
    },
    {
      title: 'FP 고유번호',
      dataIndex: 'employee_id',
      key: 'employee_id',
    },
    {
      title: 'FP명',
      dataIndex: 'name',
      key: 'name',
    },
    {
      title: '시청 횟수',
      children: [
        {
          key: 'video',
          title: '영상',
          dataIndex: 'video',
          align: 'center',
          width: 100,
        },
        {
          key: 'document',
          title: 'PDF',
          dataIndex: 'document',
          align: 'center',
          width: 100,
        },
      ],
    },
    {
      title: '메인페이지 접속 수',
      dataIndex: 'main',
      key: 'main',
      width: 90,
    },
  ];

  const selectTxt = [
    {
      placeholder: '사업본부 선택',
      txt: 'l2_name',
    },
    {
      placeholder: '권역 선택',
      txt: 'region',
    },
    {
      placeholder: '지역단 선택',
      txt: 'area',
    },
    {
      placeholder: '지점 선택',
      txt: 'branch',
    },
  ];

  return (
    <StatisticsDetailWrap>
      {loading && <PageLoader />}
      <SearchBar
        searchDate={{ title: '검색일자', visible: true }}
        handleSearch={handleSearch}
        noSearchFromToday={true}
        onReset={handleReset}
        customFormItems={[
          {
            key: 'team',
            name: 'team',
            title: '소속',
            render: () => {
              return (
                <SelectWrap>
                  {selectTxt.map((item) => (
                    <Select
                      key={item.txt}
                      placeholder={item.placeholder}
                      disabled={typeValue === '전체' && true}
                      value={selectValue[item.txt]}
                      onChange={(e) => {
                        onSelectChange(e, item.txt);
                      }}
                    >
                      {setOptionData(item.txt)}
                    </Select>
                  ))}
                </SelectWrap>
              );
            },
          },
          {
            key: 'type',
            name: 'type',
            title: '채널 구분',
            render: () => {
              return (
                <Radio.Group onChange={onTypeChange} value={typeValue}>
                  <Radio value="전체">전체</Radio>
                  <Radio value="한화생명금융서비스">한화생명금융서비스</Radio>
                  <Radio value="한화라이프랩">한화라이프랩</Radio>
                  <Radio value="GA">GA</Radio>
                </Radio.Group>
              );
            },
          },
          {
            key: 'fp',
            name: 'fp',
            title: 'FP 고유번호',
            render: () => {
              return (
                <Input
                  value={searchFp}
                  onChange={(e) => {
                    setSearchFp(e.target.value);
                  }}
                />
              );
            },
          },
        ]}
      />
      <TableWrap>
        <PaginationTable
          noAsync
          loading={isValidating}
          rowKey="employee_id"
          customLeft={<Button onClick={handleXlsxDownload}>Excel 다운로드</Button>}
          dataSource={data || []}
          columns={columns}
          pagination={pagination}
          onChangePageSize={onChangePageSize}
        />
      </TableWrap>
    </StatisticsDetailWrap>
  );
};
