// base
import React, { useRef, useState, useEffect } from 'react';

// libraries
import ReactQuill, { Quill } from 'react-quill';
import 'react-quill/dist/quill.snow.css';

// style
import './index.css';

const fontSize = Quill.import('attributors/style/size');

fontSize.whitelist = ['14px', '16px', '18px', '20px', '22px', '24px', '28px', '32px'];

Quill.register(fontSize, true);

interface ToolbarProps {
  editor: Quill | null;
}

function Toolbar(props: ToolbarProps) {
  return (
    <div id="toolbar">
      <span className="ql-formats" style={{ marginRight: 0 }}>
        <button className="ql-bold" />
        <button className="ql-italic" />
        <button className="ql-underline" />
        <button className="ql-strike" />
        <button className="ql-blockquote" />
      </span>
      <span className="ql-formats" style={{ marginRight: 0 }}>
        <select className="ql-size" defaultValue={fontSize.whitelist[0]}>
          {fontSize.whitelist.map((value: string) => (
            <option key={value} value={value}>
              {value}
            </option>
          ))}
        </select>
      </span>
      <span className="ql-formats" style={{ marginRight: 0 }}>
        <select className="ql-color" />
        <select className="ql-background" />
      </span>
      <span className="ql-formats" style={{ marginRight: 0 }}>
        <select className="ql-align" />
        <button className="ql-clean" />
      </span>
      <span className="ql-formats" style={{ marginRight: 0 }}>
        <button className="ql-code-block" />
      </span>
    </div>
  );
}

interface TextEditorProps {
  value?: string;
  onChange?: (content: string | undefined) => void;
}

export const Editor = React.forwardRef<HTMLDivElement, TextEditorProps>((props, ref) => {
  const { value, onChange } = props;
  const [editor, setEditor] = useState<Quill | null>(null);
  const quillRef = useRef<ReactQuill>(null);

  const handleChange = (content: string | undefined) => {
    if (content === '<p><br></p>') {
      content = undefined;
    }

    if (onChange) {
      onChange(content);
    }
  };

  useEffect(() => {
    if (quillRef.current) {
      setEditor(quillRef.current.getEditor());
    }
  }, []);

  return (
    <div className="text-editor" ref={ref}>
      <Toolbar editor={editor} />
      <ReactQuill
        placeholder="상세 내용을 입력해주세요"
        ref={quillRef}
        style={{ height: 200, overflowY: 'scroll' }}
        theme="snow"
        modules={{
          toolbar: {
            container: '#toolbar',
          },
        }}
        value={value ? value : ''}
        onChange={handleChange}
      />
    </div>
  );
});
