"use strict";

module.exports = function (Document) {
  const formidable = require("formidable");
  const fs = require("fs");
  const { execSync } = require("child_process");
  const COS = require("cos-nodejs-sdk-v5");
  const cos = new COS({
    SecretId: process.env.TENCENT_SECRETID,
    SecretKey: process.env.TENCENT_SECRETKEY,
  });

  Document.observe("before save", function (context, next) {
    const updatedDate = new Date();
    const _instance = context.instance;

    if (_instance) {
      _instance.created = updatedDate;
      _instance.updated = updatedDate;
    } else {
      if (!context.currentInstance) {
        return next();
      } else {
        context.data.updated = updatedDate;
      }
    }

    return next();
  });

  Document.beforeRemote("find", function (context, modelInstance, next) {
    if (!context.args.filter) {
      context.args.filter = {};
    }

    let _where = {
      status: { neq: "inactive" },
    };

    if (context.args.filter.where) {
      if (context.args.filter.where.and) {
        context.args.filter.where.and.push(_where);
      } else {
        context.args.filter.where = {
          and: [context.args.filter.where, _where],
        };
      }
    } else {
      context.args.filter["where"] = _where;
    }

    return next();
  });

  Document.beforeRemote("count", function (context, modelInstance, next) {
    if (!context.args.where) {
      context.args.where = {};
    }

    let _where = {
      status: { neq: "inactive" },
    };

    if (context.args.where) {
      if (context.args.where.and) {
        context.args.where.and.push(_where);
      } else {
        context.args.where = {
          and: [context.args.where, _where],
        };
      }
    } else {
      context.args.where = _where;
    }

    return next();
  });

  Document.uploadFile = (req, res, cb) => {
    const form = formidable.IncomingForm();

    form.parse(req, function (err, fields, files) {
      if (err) {
        return cb(err);
      }

      if (!fields.data) {
        throw "no data";
      }

      const data = JSON.parse(fields.data);
      const file = files["file"];
      const fileName = file?.name;
      const randomStr = Math.random().toString(16).slice(2);

      return new Promise((resolve, reject) => {
        cos.putObject(
          {
            Bucket: process.env.COS_BUCKET,
            Region: process.env.COS_REGION,
            Key: `${process.env.ENV}/${randomStr}_${fileName}`,
            Body: fs.createReadStream(file.path),
          },
          (err, data) => {
            if (err) return reject(err);

            return resolve(`${process.env.ENV}/${randomStr}_${fileName}`);
          }
        );
      })
        .then((key) => {
          if (!data.id) {
            return Document.create({
              ...data,
              fileKey: key,
            });
          }

          return Document.update({ id: data.id }, { ...data, fileKey: key });
        })
        .then((result) => {
          return cb(null, result);
        })
        .catch((err) => {
          return cb(err);
        });
    });
  };

  Document.readFile = (id, fileName, fileKey, res, cb) => {
    if (fileKey) {
      cos.getObject(
        {
          Bucket: process.env.COS_BUCKET,
          Region: process.env.COS_REGION,
          Key: fileKey,
        },
        function (err, data) {
          if (err) {
            return cb(err);
          }
          return cb(null, Buffer.from(data.Body).toString("base64"));
        }
      );
    } else {
      const path = `${process.env.PWD}/document/${id}_${fileName}`;
      fs.readFile(path, { encoding: "base64" }, (err, data) => {
        if (err) {
          return cb("no file");
        }
        return cb(null, data);
      });
    }
  };
};
