import { message } from 'antd';
import axios, { AxiosInstance, AxiosRequestConfig, AxiosError } from 'axios';
import { ErrorCode, errorCodeMethods } from 'types/error';
import { storageService, STORAGE_SESSION_ACT } from './storage';

export const extractErrorMsg = (error: AxiosError) => {
  if (!error.response) {
    return '서버에 접속할 수 없습니다';
  } else {
    return error.response.data.error.message || '에러 발생';
  }
};

class AxiosInstanceCreator {
  private instance: AxiosInstance;

  constructor(config?: AxiosRequestConfig) {
    this.instance = axios.create(config);
    this.interceptors();
  }

  interceptors() {
    this.instance.interceptors.request.use((config) => {
      const act = storageService.session.getItem(STORAGE_SESSION_ACT);

      if (!config.params) {
        if (act) {
          config['params'] = {
            access_token: act,
          };
        }
      } else {
        if (act) {
          Object.assign(config.params, {
            access_token: act,
          });
        }
      }

      if (!config.headers['access_token']) {
        if (act) {
          Object.assign(config.headers, {
            access_token: act,
          });
        }
      }

      return config;
    });

    this.instance.interceptors.response.use(
      (res) => {
        if (res.data.errCode) {
          throw errorCodeMethods.getErrorMsg(res.data.errCode);
        }

        return res;
      },
      (error) => {
        if (axios.isCancel(error)) {
          throw error;
        } else {
          if (error?.response?.data?.error?.code === ErrorCode.AUTHORIZATION_REQUIRED) {
            return message.error('서버 오류 관리자에게 문의주세요');
          }

          throw extractErrorMsg(error);
        }
      }
    );
  }

  create() {
    return this.instance;
  }
}

export default AxiosInstanceCreator;
